<?php

defined('BASEPATH') or exit('No direct script access allowed');

/*
Module Name: Login Attempt Guard
Description: The number of failed login attempts is limited to prevent unauthorized access and brute-force attacks.
Version: 1.0.0
Requires at least: 2.3.*
Author: Hopperstack
Author URI: https://codecanyon.net/user/hopperstack
*/

define('LOGIN_ATTEMPT_GUARD_MODULE_NAME', 'login_attempt_guard');
hooks()->add_action('admin_init', 'login_attempt_guard_init_menu_items');
hooks()->add_action('failed_login_attempt', 'failed_login_attempts');
hooks()->add_action('non_existent_user_login_attempt', 'check_non_existent_user_login_attempt');
hooks()->add_action('before_staff_login', 'check_before_staff_login');
hooks()->add_action('before_client_login', 'check_before_client_login');


/**
 * Register language files, must be registered if the module is using languages
 */
register_language_files(LOGIN_ATTEMPT_GUARD_MODULE_NAME, [LOGIN_ATTEMPT_GUARD_MODULE_NAME]);


$CI = & get_instance();
$CI->load->helper(LOGIN_ATTEMPT_GUARD_MODULE_NAME . '/login_attempt_guard');

/**
 * Init login attempt guard module menu items in setup in admin_init hook
 * @return null
 */
function login_attempt_guard_init_menu_items()
{
    if (is_admin()) {   
        $CI = &get_instance();
    
            $CI->app_menu->add_sidebar_menu_item('login_attempt_guard', [
                'name'     => _l('login_attempt_guard'),
                'icon'     => 'fa fa-shield',
                'position' => 40,
            ]);
            $CI->app_menu->add_sidebar_children_item('login_attempt_guard', [
                'slug'     => 'locked_list',
                'name'     => _l('locked_list'),
                'href'     => admin_url('login_attempt_guard/locked_list'),
                'position' => 1,
            ]);
            $CI->app_menu->add_sidebar_children_item('login_attempt_guard', [
                'slug'     => 'manage',
                'name'     => _l('setting'),
                'href'     => admin_url('login_attempt_guard/manage'),
                'position' => 5,
            ]);
    
    }
}
/** 
 * Handle module activation tasks (e.g., database setup)
*/
register_activation_hook(LOGIN_ATTEMPT_GUARD_MODULE_NAME, 'login_attempt_guard_module_activation_hook');

function login_attempt_guard_module_activation_hook()
{
    $CI = &get_instance();
    require_once(__DIR__ . '/install.php');
}

function failed_login_attempts($data) {

    // Check if the login attempt guard functionality is enabled
    if (option_exists('login_attempt_guard_functionality') && get_option('login_attempt_guard_functionality') == 1) {
        $CI = &get_instance();  
        $ip = $CI->input->ip_address();
        $user_agent = substr($CI->input->user_agent(), 0, 149);
        $how_many_attempts = get_option('allow_retries');
        $minute_lockout = get_option('minute_lockout');
        $user = $data['user'];
        $table = db_prefix() . 'login_attempts_guard';
        $type = $data['is_staff_member'] ? 'staff' : 'client';
        $current_time = date('Y-m-d H:i:s');

        // Check if a record for this IP already exists
        if (check_is_record_existing($ip)) {
            $record = $CI->db->where('ip', $ip)->get($table)->row();

            if ($record) {
                if ($record->attempt_count >= $how_many_attempts) {
                    if ($current_time <= $record->locked_until) {
                        // Account is locked, calculate remaining time
                        $remaining_minutes = display_calculate_remaining_time($record->locked_until);
                        set_alert('danger', 'You have exceeded the maximum number of login attempts.<br>Please try again after ' . $remaining_minutes.'.');

                        // Redirect to appropriate authentication page
                        $redirect_url = $data['is_staff_member'] ? admin_url('authentication') : site_url('authentication');
                        redirect($redirect_url);
                    } else {
                        // Lockout period is over, reset the lockout
                        reset_login_attempts($ip);
                    }
                } else if ($record->attempt_count >= ($how_many_attempts - 1)) {
                    // Lock the account for the specified lockout duration
                    $locked_until = date('Y-m-d H:i:s', strtotime('+' . $minute_lockout . ' minutes'));
                    update_login_attempts($ip, $record->attempt_count + 1, $locked_until);
                    $remaining_minutes = display_calculate_remaining_time($locked_until);
                    set_alert('danger', 'You have exceeded the maximum number of login attempts.<br>Please try again after ' . $remaining_minutes.'.');
                } else {
                    // Increment the attempt count and display remaining attempts
                    $attempt_count = $record->attempt_count + 1;
                    update_login_attempts($ip, $attempt_count, NULL);
                    $remaining_attempts = $how_many_attempts - $attempt_count;
                    set_alert('danger', _l('invalid_email_or_password') . '<br>' . $remaining_attempts . ' attempts remaining.');
                }
            }

        } else {
            // Insert a new record for this IP
            $data_to_insert = [
                'email_id' => $user->email,
                'attempt_count' => 1,
                'ip' => $ip,
                'useragent' => $user_agent,
                'type' => $type,
                'last_attempt_at' => $current_time,
                'date' => $current_time
            ];
            $CI->db->insert($table, $data_to_insert);

            $remaining_attempts = $how_many_attempts - 1;
            set_alert('danger', _l('invalid_email_or_password') . '<br>' . $remaining_attempts . ' attempts remaining.');
        }

        // Redirect to the appropriate authentication page after the attempt
        $redirect_url = $data['is_staff_member'] ? admin_url('authentication') : site_url('authentication');
        redirect($redirect_url);
    }
}

/**
 * Check if a record exists for the given IP address.
 * @param string $ip
 * @return bool
 */
function check_is_record_existing($ip) {
    $CI = &get_instance();
    $table = db_prefix() . 'login_attempts_guard';
    return $CI->db->where('ip', $ip)->count_all_results($table) > 0;
}

/**
 * Update login attempts count and lockout time for the given IP address.
 * @param string $ip
 * @param int $attempt_count
 * @param string|null $locked_until
 */
function update_login_attempts($ip, $attempt_count, $locked_until) {
    $CI = &get_instance();
    $table = db_prefix() . 'login_attempts_guard';
    $CI->db->set('attempt_count', $attempt_count)
           ->set('last_attempt_at', date('Y-m-d H:i:s'))
           ->set('locked_until', $locked_until)
           ->where('ip', $ip)
           ->update($table);
}

/**
 * Reset login attempts for the given IP address.
 * @param string $ip
 */
function reset_login_attempts($ip) {
    $CI = &get_instance();
    $table = db_prefix() . 'login_attempts_guard';
    $CI->db->set('attempt_count', 0)
           ->set('locked_until', NULL)
           ->where('ip', $ip)
           ->update($table);
}

/**
 * Calculate remaining lockout time.
 * @param string $locked_until
 * @return int Remaining minutes
 */
function calculate_remaining_time($locked_until) {
    $current_time = strtotime(date('Y-m-d H:i:s'));
    $lockout_time = strtotime($locked_until);
    return ceil(($lockout_time - $current_time) / 60);
}


function check_non_existent_user_login_attempt($data){
    // Check if the login attempt guard functionality is enabled
    if (option_exists('login_attempt_guard_functionality') && get_option('login_attempt_guard_functionality') == 1) {
        $CI = &get_instance();  
        $ip = $CI->input->ip_address();
        $user_agent = substr($CI->input->user_agent(), 0, 149);
        $how_many_attempts = get_option('allow_retries');
        $minute_lockout = get_option('minute_lockout');
        $user = $data['email'];
        $table = db_prefix() . 'login_attempts_guard';
        $type = $data['is_staff_member'] ? 'staff' : 'client';
        $current_time = date('Y-m-d H:i:s');

        // Check if a record for this IP already exists
        if (check_is_record_existing($ip)) {
            $record = $CI->db->where('ip', $ip)->get($table)->row();

            if ($record) {
                if ($record->attempt_count >= $how_many_attempts) {
                    if ($current_time <= $record->locked_until) {
                        // Account is locked, calculate remaining time
                        $remaining_minutes = display_calculate_remaining_time($record->locked_until);
                        set_alert('danger', 'You have exceeded the maximum number of login attempts.<br>Please try again after ' . $remaining_minutes.'.');

                        // Redirect to appropriate authentication page
                        $redirect_url = $data['is_staff_member'] ? admin_url('authentication') : site_url('authentication');
                        redirect($redirect_url);
                    } else {
                        // Lockout period is over, reset the lockout
                        reset_login_attempts($ip);
                    }
                } else if ($record->attempt_count >= ($how_many_attempts - 1)) {
                    // Lock the account for the specified lockout duration
                    $locked_until = date('Y-m-d H:i:s', strtotime('+' . $minute_lockout . ' minutes'));
                    update_login_attempts($ip, $record->attempt_count + 1, $locked_until);
                    $remaining_minutes = display_calculate_remaining_time($locked_until);
                    set_alert('danger', 'You have exceeded the maximum number of login attempts.<br>Please try again after ' . $remaining_minutes.'.');
                } else {
                    // Increment the attempt count and display remaining attempts
                    $attempt_count = $record->attempt_count + 1;
                    update_login_attempts($ip, $attempt_count, NULL);
                    $remaining_attempts = $how_many_attempts - $attempt_count;
                    set_alert('danger', _l('invalid_email_or_password') . '<br>' . $remaining_attempts . ' attempts remaining.');
                }
            }

        } else {
            // Insert a new record for this IP
            $data_to_insert = [
                'email_id' => $user->email,
                'attempt_count' => 1,
                'ip' => $ip,
                'useragent' => $user_agent,
                'type' => $type,
                'last_attempt_at' => $current_time,
                'date' => $current_time
            ];
            $CI->db->insert($table, $data_to_insert);

            $remaining_attempts = $how_many_attempts - 1;
            set_alert('danger', _l('invalid_email_or_password') . '<br>' . $remaining_attempts . ' attempts remaining.');
        }

        // Redirect to the appropriate authentication page after the attempt
        $redirect_url = $data['is_staff_member'] ? admin_url('authentication') : site_url('authentication');
        redirect($redirect_url);
    }
}

function check_before_staff_login($data){
 // Check if the login attempt guard functionality is enabled
    if (option_exists('login_attempt_guard_functionality') && get_option('login_attempt_guard_functionality') == 1) {
        $CI = &get_instance();  
        $ip = $CI->input->ip_address();
        $how_many_attempts = get_option('allow_retries');
        $table = db_prefix() . 'login_attempts_guard';
        $type = 'staff';
        $current_time = date('Y-m-d H:i:s');

        // Check if a record for this IP already exists
        if (check_is_record_existing($ip)) {
            $record = $CI->db->where('ip', $ip)->get($table)->row();

            if ($record) {
                if ($record->attempt_count >= $how_many_attempts) {
                    if ($current_time <= $record->locked_until) {
                        // Account is locked, calculate remaining time
                        $remaining_minutes = display_calculate_remaining_time($record->locked_until);
                        set_alert('danger', 'You have exceeded the maximum number of login attempts.<br>Please try again after ' . $remaining_minutes.'.');

                        // Redirect to appropriate authentication page
                        $redirect_url = $type == 'staff' ? admin_url('authentication') : site_url('authentication');
                        redirect($redirect_url);
                    }
                } 
            }

        } 

    }
}
function before_client_login($data){
     // Check if the login attempt guard functionality is enabled
     if (option_exists('login_attempt_guard_functionality') && get_option('login_attempt_guard_functionality') == 1) {
        $CI = &get_instance();  
        $ip = $CI->input->ip_address();
        $how_many_attempts = get_option('allow_retries');
        $table = db_prefix() . 'login_attempts_guard';
        $type = 'client';
        $current_time = date('Y-m-d H:i:s');

        // Check if a record for this IP already exists
        if (check_is_record_existing($ip)) {
            $record = $CI->db->where('ip', $ip)->get($table)->row();

            if ($record) {
                if ($record->attempt_count >= $how_many_attempts) {
                    if ($current_time <= $record->locked_until) {
                        // Account is locked, calculate remaining time
                        $remaining_minutes = display_calculate_remaining_time($record->locked_until);
                        set_alert('danger', 'You have exceeded the maximum number of login attempts.<br>Please try again after ' . $remaining_minutes.'.');

                        // Redirect to appropriate authentication page
                        $redirect_url = $type == 'staff' ? admin_url('authentication') : site_url('authentication');
                        redirect($redirect_url);
                    }
                } 
            }

        } 

    }
}